/**
* \file: AlsaAudioInImpl.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: CarPlay
*
* \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
*
* \copyright (c) 2013-2014 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <memory.h>
#include <pthread.h>
#include <sys/prctl.h>
#include <limits.h>
#include <adit_logging.h>
#include <dipo_macros.h>
#include "AlsaAudioIn.h"
#include "AlsaAudioInImpl.h"
#include "AlsaConfiguration.h"

using namespace std;

LOG_IMPORT_CONTEXT(cply);

namespace adit { namespace carplay
{

AlsaAudioIn::Impl::Impl()
{
    config = nullptr;
    sink = nullptr;

    processThreadId = 0;

    periodMilli = 0;
    periodSamples = 0;
    appendEmptyPeriods = 0;

    running = false;
}

AlsaAudioIn::Impl::~Impl()
{
    /* PRQA: Lint Message 1506: Stop is not expected to be further overridden */
    /*lint -save -e1506*/
    Stop();
    /*lint -restore*/
}

bool AlsaAudioIn::Impl::Initialize(const IConfiguration& inConfig,
        IAudioInSink& inSink)
{
    config = &inConfig;
    sink = &inSink;

    bool verboseLogging = AlsaConfiguration::Instance().IsVerboseLogging(inConfig);
    bool alsaLogging = AlsaConfiguration::Instance().IsAlsaLogging(inConfig);

    alsa = move(unique_ptr<AlsaAudioCommon>(new AlsaAudioCommon(alsaLogging, verboseLogging, inConfig)));
    return true;
}

bool AlsaAudioIn::Impl::Prepare(AudioFormatStruct inFormat, const std::string& inAudioType)
{
	int32_t buffer_periods = -1;
	int32_t silence_ms = -1;
	int32_t inittout_ms = -1;

    (void)inFormat;

    if (config == nullptr)
    {
        LOG_ERROR((cply, "AlsaAudioOut is not initialized"));
        return false;
    }

    format = inFormat;

    if (format.BitsPerChannel != 24 && format.BitsPerChannel != 16)
    {
        LOG_ERROR((cply, "AlsaAudioIn does not support %d bit audio", format.BitsPerChannel));
        return false; /* ========== leaving function ========== */
    }

    // numbers of periods to append after end-of-stream
    appendEmptyPeriods = config->GetNumber("alsa-audio-out-append-empty-periods", 0);

    if (!AlsaConfiguration::Instance().GetDeviceSettings(*config, AlsaConfiguration::Channel_MainIn,
            inAudioType, inFormat, deviceName /* out */, periodMilli /* out */,
            buffer_periods /* out */, silence_ms /* out */, inittout_ms /* out */))
    {
        LOG_ERROR((cply, "failed to get audio device settings"));
        return false; /* ========== leaving function ========== */
    }

    alsa->SetDir( SND_PCM_STREAM_CAPTURE );
    alsa->SetRate( format.SampleRate );
    alsa->SetChannels( format.Channels );
    alsa->SetIdealPeriodMs( periodMilli );
    if (buffer_periods >= 0)
        alsa->SetBufferIdealPeriods( buffer_periods );
    if (silence_ms >= 0)
        alsa->SetPrefillMs ( silence_ms );
    if (inittout_ms >= 0)
        alsa->SetInitTout ( inittout_ms );

    switch (format.BitsPerChannel) {
        case 16 : {
            alsa->SetFormat( SND_PCM_FORMAT_S16_LE );
            break;
        }
        case 24 : {
            alsa->SetFormat( SND_PCM_FORMAT_S24_3LE );/* TODO check if this is correct 24bit data in 3bytes or if S24_LE 24bit in 4bytes is correct */
            break;
        }
        default :
            LOG_ERROR((cply, "AlsaAudioOut does not support %d bits per channel\n", format.BitsPerChannel));
            return false;
    }

    if (0 != alsa->SetupPCM((char*)deviceName.c_str())) {
        return false;
    }

    periodSamples = alsa->GetIdealPeriodMs();

    LOGD_DEBUG((cply, "main audio in: device=%s, period=%dms",
            deviceName.c_str(), periodMilli));
    return true;
}

bool AlsaAudioIn::Impl::Start()
{
    running = true;

    if (0 != pthread_create(&processThreadId, nullptr, process, this))
    {
        LOG_ERROR((cply, "could not create audio out thread"));
        return false;
    }

    LOGD_DEBUG((cply, "main audio in started"));
    return true;
}

void AlsaAudioIn::Impl::Stop()
{
    bool wasRunning = running;
    running = false;

    /* Abort streaming to avoid recover retries (long blocking) */
    alsa->AbortStreaming();

    if (processThreadId != 0)
    {
        pthread_join(processThreadId, nullptr);
        processThreadId = 0;
    }

    if (wasRunning)
        LOGD_DEBUG((cply, "main audio in stopped"));
}

void* AlsaAudioIn::Impl::process(void* inData)
{
    (void)inData;

    auto me = static_cast<AlsaAudioIn::Impl*>(inData);
    dipo_return_value_on_invalid_argument(cply, me == nullptr, nullptr);

    // set thread name
    prctl(PR_SET_NAME, "AlsaMainAudioIn", 0, 0, 0);

    // set thread prioirty
    me->alsa->SetThreadPriority("AlsaMainAudioIn");

    uint64_t sampleNumber = 0;

    while (me->running)
    {
        const int bufferSize = me->periodSamples *
                (me->format.BitsPerChannel / 8) * me->format.Channels;
        char buffer[bufferSize];

        Samples samples;
        samples.DataPtr = buffer;
        samples.Length = sizeof(buffer);

        // TODO
        {
            // default fallback to samples to were pushed to the pipeline
            samples.TimeStamp = sampleNumber;
        }

        int err = 0;
        if ((err = me->alsa->ReadWrite(buffer, bufferSize)) < 0)
        {
            LOG_ERROR((cply, "write to RTS failed with %d", err));
            break;
        }

        me->sink->Write(samples);
        sampleNumber += me->periodSamples;
    }

    me->alsa->StopStreaming();
    me->alsa->ReleasePCM();

    return nullptr;
}

} } // namespace adit { namespace carplay
